//! AX.25 datagram protocol
/**
 * The AAU-CubeSat AX.25 protocol, UI laye */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "buffers.h"
#include "ax25.h"
#include "crc.h"

/* ************************************************************************ */

char TxFrameBufferIF[300];

/* ************************************************************************ */

const unsigned char AX25_FRAME_HEADER[16] = {
  'U' << 1, 'I' << 1, ' ' << 1, ' ' << 1, ' ' << 1, ' ' << 1, 0x60, \
  'G' << 1, '7' << 1, 'O' << 1, 'C' << 1, 'D' << 1, ' ' << 1, 0x61, 0x03, 0xf0};

// aa 92 40 40 40 40 60 8e 6e 9e 86 88 40 61 03 f0

/* ************************************************************************ */
int encode_frame(unsigned char *out, unsigned char *in, int in_length, char FrameType, unsigned char *callsign){
  int outpos = 0;
  int ax25f_length;
  
  ax25f_length = sizeof(AX25_FRAME_HEADER);

  //calc the total length of the frame contens
  // /*info_length = */ax25f_length + in_length; // This did nothing ...
  //copy the callsign to the buffer, skip over the flag
  memcpy(out,AX25_FRAME_HEADER,ax25f_length);
  out[7] = (unsigned char) (callsign[0] << 1) & 0xfe;
  out[8] = (unsigned char) (callsign[1] << 1) & 0xfe;
  out[9] = (unsigned char) (callsign[2] << 1) & 0xfe;
  out[10] = (unsigned char) (callsign[3] << 1) & 0xfe;
  out[11] = (unsigned char) (callsign[4] << 1) & 0xfe;
  out[12] = (unsigned char) (callsign[5] << 1) & 0xfe;

  //   printf("ax25_head: ");memdump(out, ax25f_length);
  //keep track of how much we've written to the buffer 
  outpos = ax25f_length;
  
  if(FrameType == Command){//Set the frame type
    out[7] |= 0x80;//Set the destination SSID command bit
    out[14] &= 0x7f;//Clear the source SSID command bit
  }
  else{
    out[14] |= 0x80;//Set the source SSID command bit
    out[7] &= 0x7f;             //Clear the destination SSID command bit
  }
  memcpy((out + outpos ),in,in_length);//copy the actual data
  outpos += in_length;
  return outpos;//return the length of the complete package
}


/* ************************************************************************ */
//this function will remove all the AX.25 header and transport encoding on return 
//it will give a zero length to indicate a rejected frame
int decode_frame(unsigned char *out, unsigned char *in, int in_length, unsigned char *callsign)
{
  int length = 0;

  callsign[0] = (unsigned char) (in[7] >> 1) & 0x7f;
  callsign[1] = (unsigned char) (in[8] >> 1) & 0x7f;;
  callsign[2] = (unsigned char) (in[9] >> 1) & 0x7f;;
  callsign[3] = (unsigned char) (in[10] >> 1) & 0x7f;;
  callsign[4] = (unsigned char) (in[11] >> 1) & 0x7f;;
  callsign[5] = (unsigned char) (in[12] >> 1) & 0x7f;

  if(in_length < 16)
    return -1;
  
  //the length of the frame information, is 
  length = in_length - sizeof(AX25_FRAME_HEADER);
  //the unstuffed length minus the first flag and the crc (3)
  memcpy(out,(in + sizeof(AX25_FRAME_HEADER)),length);
  return (length);
}
/* ************************************************************************ */



// Always remember that out should be set to zeros before using this function. 
int bit_stuff(unsigned char *out, unsigned char *in, int in_length, int *out_length){
  int num_bits_in = in_length * 8;
  int nbo = 0;
  char bit;
  //char temp[1000];
  int onebits = 0;
  int i;

  for(i=0;i<num_bits_in;i++){
    // Read one bit
    bit = (unsigned char)(*(in + i/8) >> (i&7)) & 0x01;
    //sprintf(temp,"bit%d: %d\r\n",i, bit);
    //strcat(out, temp);
    if(bit == 1)
      onebits++;
    else
      onebits = 0;

    // Insert the bit in the output data
    *(out+nbo/8) = (*(out+nbo/8) & (0xff^(1 << (nbo&7)))) | (bit << (nbo&7));
    nbo++;

    if(onebits == 5){
      // Insert a zero-bit
      // debug_printl("Inserting zero bit at %d ",  nbo);
      *(out+nbo/8) = *(out+nbo/8) & (0xff^(1 << (nbo&7)));
      nbo++;

      // Reset onebits counter
      onebits = 0;
    }
  }

  *out_length = nbo / 8;
  // Check for extra byte
  if(*out_length*8 < nbo)
    *out_length += 1;

  return 0;
}


int bit_unstuff(unsigned char *out, unsigned char *in, int in_length, int *out_length){
  int num_bits_in = in_length * 8;
  int nbo = 0;
  char bit;
  int onebits = 0;
  //char temp[100];
  int i;

  for(i=0;i<num_bits_in;i++){
    // Read one bit
    bit = (unsigned char)(*(in + i/8) >> (i&7)) & 0x01 ;
    if((i >= 15*8) && (i <= 18*8)){
      //sprintf(temp,"bit%d: %d\r\n",i, bit);
      //strcat(dbg, temp);
    }

    if(bit == 1)
      onebits++;
    else
      onebits = 0;

    if(onebits == 5){
      //strcat(dbg, "onebits limit reached\r\n");
      // Don't read next bit
      i++;

      // Reset onebits counter
      onebits = 0;
    }

    // Insert the bit in the output data
    *(out+nbo/8) = (*(out+nbo/8) & (0xff^(1 << (nbo&7)))) | (bit << (nbo&7));
    nbo++;

  }

  *out_length = nbo / 8;


  return 0;
}



int encode_ax25(unsigned char *out, unsigned char *in, int in_length, int *out_length, unsigned char rssi, unsigned char hferr, unsigned char *callsign, unsigned char *dbg){
  int length, iCrc;
  unsigned char cBuf[600];
  //char cBuf2[600];
  //char dump[2000];

  memset(cBuf, 0, 600);
  memset(out, 0, 600);

  // Create AX.25 header
  length = encode_frame(cBuf, in, in_length, 0, callsign);
  //COM_printf("Encoding frame %d",length);
  //  dump[0] = 0;
  //  memdump(dump, cBuf, length);
  //  debug_printl(dump);

  //Add the RSSI and HFERR of the last received packet
  *(cBuf+length) = rssi;
  *(cBuf+length+1) = hferr;

  length += 2;

  // Calculate and copy checksum
  iCrc = crc((char*) cBuf,length);
  memcpy(cBuf+length, (unsigned char*) &iCrc, 2);
  length += 2;
  // COM_printf("CRC: 0x%x, %d", iCrc & 0xffff, length+2);
  //dump[0] = 0;
  //memdump(dump, cBuf, length+2);
  //debug_printl(dump);

  // Bit Stuff
  bit_stuff(out+1, cBuf, length, &length);
  //COM_printf("Bit stuffing %d", length+1);
  //dump[0] = 0;
  //memdump(dump, out, length+1);
  //debug_printl(dump);

  // Add AX.25 header and tail
  *out = 0x7e;
  *(out+length+1) = 0x7e;

  // One extra byte added to the length in order to keep TX up until hwole packet have been transmitted.
  *out_length = length + 3;
  //COM_printf("Adding head and tail %d", *out_length);
  //dump[0] = 0;
  //memdump(dump, out, *out_length);
  //debug_printl(dump);

  return 0;
}

int decode_ax25(unsigned char *out, unsigned char *in, int in_length, int *out_length, unsigned char *rssi, unsigned char *hferr, unsigned char *callsign){
  int length;
  short iCrc, iCrc2;
  unsigned char cBuf[600];
  unsigned char cBuf2[600];
  //char str[400];

  memset(cBuf, 0, 600);
  memset(cBuf2, 0, 600);

  // Bit unstuff frame (minus header and tail)
  bit_unstuff(cBuf, in+1, in_length-2, &length);

  //memcpy(out, cBuf, length);
  //*out_length = length;

  // Calculate checksum
  iCrc = crc((char*) cBuf,length-2);

  // Decode the AX25 frame
  length = decode_frame(cBuf2, cBuf, length, callsign);

  //sprintf(dbg, "Length: %d\n", length);

  if(length < 4 || length > 300){  //FIX: Changed lower boundary to 4 as it causes problems with the memcpy further down if the length is 0 or 1 because 2 is subtracted.
    *out_length = 0;
    return -2;
  }

  // Get rid of checksum, rssi and hferr
  length = length - 4;
  memcpy(out, cBuf2, length);
  *out_length = length;

  iCrc2 = 0;
  memcpy(&iCrc2, cBuf2+length+2, 2);

  *rssi = cBuf2[length];
  *hferr = cBuf2[length+1];

  //printf("checksums: 0x%x   0x%x\n", iCrc, iCrc2);
  //strcat(dbg,str);

  // Compare checksums
  if(iCrc != iCrc2)
    return -1;

  return 0;
}

int check_ax25_buffer(ring_buffer *buf, unsigned char *ax25_data, int *ax25_len){

  int offset1 = 0, offset2 = 0;
  int bytes_in_buf, header_offset;
  int i;
  unsigned char ch1, ch2, ch3;

  *ax25_len = 0;

  // Get the number of bytes in the buffer
  bytes_in_buf = bytes_in_buffer(buf);

  // Check for length
  if(bytes_in_buf < 4)
    return 0;

  // Find the packet header (0x7e 0xaa 0x92)
  header_offset = -1;
  peekbyte(buf, &ch1, 0);
  peekbyte(buf, &ch2, 1);
  peekbyte(buf, &ch3, 2);
  //printf("1offset1: %d, ch1: 0x%02x, ch2: 0x%02x, ch3: 0x%02x\n",offset1,ch1,ch2,ch3);
  for(offset1 = 0; offset1 < bytes_in_buf-3; offset1++) {
    if((ch1 == 0x7e) && (ch2 == 0xaa) && (ch3 == 0x92)){
      header_offset = offset1;
      //printf("2offset1: %d, ch1: 0x%02x, ch2: 0x%02x, ch3: 0x%02x\n",offset1,ch1,ch2,ch3);
      break;
    }
    ch1 = ch2;
    ch2 = ch3;
    peekbyte(buf, &ch3, offset1+3);
    //printf("3offset1: %d, ch1: 0x%02x, ch2: 0x%02x, ch3: 0x%02x\n",offset1,ch1,ch2,ch3);
  }

  // Remove garbage from buffer
  if(offset1 > 0){
    printf("MODEM_API: Removing garbage from ax25 buffer: %d\n", offset1);
    for(i=0;i<offset1;i++){
      getbyte(buf, &ch1);
      printf("%02x ", ch1);
    }
    printf("\n");
  }

  // Return if no valid header was found
  if(header_offset == -1){
    //printf("Header not found!\n");
    return 1; 
  }
  else{ // If the header was found the garbage has been removed and the header is now at 0.
    //printf("Header found!\n");
    offset1 = 0;
  }

  // Get the number of bytes in the buffer now
  bytes_in_buf = bytes_in_buffer(buf);

  // Find the tail of the packet
  for(offset2 = 1; offset2 < bytes_in_buf; offset2++) {
    peekbyte(buf, &ch1, offset2);
    if(ch1 == 0x7e){
      //printf("Tail found!\n");
      break;
    }
  }

  // Return if tail was not found
  if(ch1 != 0x7e){
    //printf("Tail not found!\n");
    return 2;
  }

  // If offset 2 has been found we have a packet and we read the data from the buffer
  if(offset2 > 0){
    *ax25_len = offset2+1;
    buf_read(buf, ax25_data, *ax25_len);
    //printf("AX25 dump:\n");
    //memdump((unsigned char*) ax25_data, offset2+1);
  }
  return 3;
}

